<?php


namespace Gfxpulse\Translation;

use Silex\Application;
use Symfony\Component\Config\ConfigCache;
use Symfony\Component\Translation\Loader\ArrayLoader;
use Symfony\Component\Translation\Loader\XliffFileLoader;
use Symfony\Component\Translation\Loader\YamlFileLoader;
use Symfony\Component\Translation\Translator as BaseTranslator;
use Symfony\Component\Translation\MessageSelector;

/**
 * Translator that gets the current locale from the Silex application.
 */
class Translator extends BaseTranslator
{
    protected $app;
    protected $loaderIds;

    public function __construct(Application $app, MessageSelector $selector, array $options = array())
    {
        $this->app = $app;
        $this->loaderIds = array();

        $this->options = array(
            'cache_dir' => null,
            'debug'     => false,
        );

        // check option names
        if ($diff = array_diff(array_keys($options), array_keys($this->options))) {
            throw new \InvalidArgumentException(sprintf('The Translator does not support the following options: \'%s\'.', implode('\', \'', $diff)));
        }

        $this->options = array_merge($this->options, $options);

        parent::__construct(null, $selector);
    }

    public function getLocale()
    {
        return $this->app['locale'];
    }

    protected function loadCatalogue($locale)
    {
        if (isset($this->catalogues[$locale])) {
            return;
        }

        // ak je cache_dir == null, tak len vtedy umoznit pridat Loadery
        if (null === $this->options['cache_dir']) {
            $this->initialize();

            return parent::loadCatalogue($locale);
        }

        $cache = new ConfigCache($this->options['cache_dir'].'/catalogue.'.$locale.'.php', $this->options['debug']);
        if (!$cache->isFresh()) {
            $this->initialize();

            parent::loadCatalogue($locale);

            $fallbackContent = '';
            $current = '';
            foreach ($this->computeFallbackLocales($locale) as $fallback) {
                $fallbackSuffix = ucfirst(str_replace('-', '_', $fallback));

                $fallbackContent .= sprintf(<<<EOF
\$catalogue%s = new MessageCatalogue('%s', %s);
\$catalogue%s->addFallbackCatalogue(\$catalogue%s);


EOF
                    ,
                    $fallbackSuffix,
                    $fallback,
                    var_export($this->catalogues[$fallback]->all(), true),
                    ucfirst(str_replace('-', '_', $current)),
                    $fallbackSuffix
                );
                $current = $fallback;
            }

            $content = sprintf(<<<EOF
<?php

use Symfony\Component\Translation\MessageCatalogue;

\$catalogue = new MessageCatalogue('%s', %s);

%s
return \$catalogue;

EOF
                ,
                $locale,
                var_export($this->catalogues[$locale]->all(), true),
                $fallbackContent
            );

            $cache->write($content, $this->catalogues[$locale]->getResources());

            return;
        }
        $this->catalogues[$locale] = include $cache;
    }

    public function addLoaderId($id) {
        $this->loaderIds[] = $id;
    }

    protected function initialize()
    {
        foreach ($this->loaderIds as $id) {
            switch ($id) {
                case 'array':
                    $this->addLoader($id, new ArrayLoader());
                    break;
                case 'xliff':
                    $this->addLoader($id, new XliffFileLoader());
                    break;
                case 'yaml':
                    $this->addLoader($id, new YamlFileLoader());
                    break;
            }
        }
    }
}
